% MATLAB script belonging to the paper: Melman, T., Weijerman, M., De Winter, J. C. F., & Abbink, D. A. (in press). Should steering settings be changed by the driver or by the vehicle itself? Human Factors.
% Written by Mark Weijerman, Timo Melman, & Joost de Winter.

clearvars -except Experiment ROAD ROAD_DISTANCE DENSITY CURVATURE questionnaires
close all;clc

% Colors used for figures
color_FL = [255 165 0]/255;
color_FH = [0 0 0]/255;
color_MI = [255 165 0]/255;
color_DI = [0 0 0]/255;

load('./Simulatordata.mat');                  % MATLAB structure with time, x and y position, travelled distance, steering wheel angle, and steering gain for all 96 trials (24 participants x 4 conditions)
load('./Road data files/ROAD.mat');           % x,y coordinates for boundary left-lane, center left-lane, lane-crossing, center right-lane, and boundary right-lane
load('./Road data files/ROAD_DISTANCE.mat');  % travelled distance on the road (m)
load('./Road data files/CURVATURE.mat');      % road curvature (1/m)
load('./Road data files/DENSITY.mat');        % 1. spawn x-position (subsequent differences are the distance between cars), 2. traffic density (computed from distances between cars), 3. x-position when the ego-vehicle encounters the car (decreases with distance travelled), 4. travelled distance when the ego-vehicle encounters the car
load('./Questionnaires.mat');                 % participants' responses to questionnaires
Roadpoints = 0.50 : 0.50 : 14345;             % distance for interpolation (m)

Order = [1 2 3 4 % 1 = FL, 2 = FH, 3 = MI, 4 = DI
    1	4	2	3
    2	4	1	3
    4	1	3	2
    1	3	4	2
    4	2	1	3
    3	2	1	4
    2	3	1	4
    2	1	4	3
    3	1	4	2
    1	4	3	2
    3	1	2	4
    1	3	2	4
    4	3	2	1
    2	4	3	1
    2	1	3	4
    4	2	3	1
    3	4	1	2
    4	3	1	2
    3	4	2	1
    2	3	4	1
    3	2	4	1
    1	2	4	3
    4	1	2	3];

NT=NaN(24,4); % 24 participants x 4 conditions
for sub = 1:24 % loop over number of participants
    for cond = 1:4 % loop over the four conditions
        shrt = simulatordata.par(sub).cond(cond).data_filt;
        [Interpolate.cond(cond).SWA(sub,:)] = interp1(shrt.Travelled_distance', shrt.SWA', Roadpoints);
        [Interpolate.cond(cond).FWA(sub,:)] = interp1(shrt.Travelled_distance', shrt.FWA', Roadpoints);
        [Interpolate.cond(cond).lateral_dev(sub,:)] = interp1(shrt.Travelled_distance', -1*shrt.lateral_dev', Roadpoints);

        % Here, calculations are done based on the recorded steering gain. 
        % The number of transitions is calculated, and a vector with 1s and
        % 0s is created, which defines the steering gain as a function of
        % time
        g=shrt.Gain'; % recorded steering gain
        g(g<1.0001)=1; g(g>1.9999)=2;g=g-1; % convert gain to a value between 0 and 1
        dg=diff(g); % detect changes in gain
        dg=medfilt1(dg,20); % apply a median filter with an interval of 20 samples (~0.24 s), within human reaction time. This ensures that very rapid fluctuations in recorded gain, so to, e.g., numeric noise, do not count as a transition

        dg(dg>0)=1; % if gain increases, then 1
        dg(dg<0)=-1; % if gain decreases, then -1

        l2h=[40+strfind(dg(40:end),[0 1]) 40+strfind(dg(40:end),[-1 1])]; % find the low to high transitions
        h2l=[40+strfind(dg(40:end),[0 -1]) 40+strfind(dg(40:end),[1 -1])]; % find the high to low transitions

        gd=zeros(size(g)); % vector of zeros
        gd(1:40)=g(40); % starting gain
        for i=1:length(l2h) % loop over low-to-high transitions
            gd(l2h(i):end)=gd(l2h(i):end)+1; % add 1 after low-to-high transition
        end
        for i=1:length(h2l) % loop over high-to-low transitions
            gd(h2l(i):end)=gd(h2l(i):end)-1; % subtract -1 after high-to-low transition
        end
        if mean(gd)<0
            gd=gd+1; % apply correction if starting gain is 1 off
        end

        % gd is now a vector with 1s for high SG, and 0s for low SG. It shows the moments the transitions were initiated
        
        NT(sub,cond)=length(l2h)+length(h2l); % Number of transitions
        [Interpolate.cond(cond).Gain(sub,:)] = interp1(shrt.Travelled_distance', gd, Roadpoints); % interpolate gain to distance
    end
end

disp('Mean number of transitions for FL, FH, MI, DI')
disp(mean(NT))
disp('SD number of transitions for FL, FH, MI, DI')
disp(std(NT))

%%
[M,Mo]=deal(NaN(24,4,3,8)); % means and means ordered

M(:,:,1,1)=questionnaires.SubjectiveEffort.Overtaking; % subjective effort (1 to 7) for the overtaking segment
M(:,:,2,1)=questionnaires.SubjectiveEffort.Straight; % subjective effort (1 to 7) for the straight segment
M(:,:,3,1)=questionnaires.SubjectiveEffort.Curves; % subjective effort (1 to 7) for the curve segment

for i=1:24 % loop over 24 participants
    for i2=1:4 % loop over the 4 experimental conditions
        for i3=1:3 % loop over the 3 driving-task segments
            switch i3
                case 1
                    segment = 3928:6928; % overtaking segment
                case 2
                    segment = 11000:19236; % straight segment
                case 3
                    segment = 19516:28690; % curve segment
            end

            M(i,i2,i3,2)=mean(abs(Interpolate.cond(i2).FWA(i,segment)),'omitnan'); % mean absolute front wheel angle (deg)
            M(i,i2,i3,3)=mean(abs((1/0.018)*diff(Interpolate.cond(i2).lateral_dev(i,segment))),'omitnan'); % mean absolute lateral velocity (deg/s); car travels 27.7777 m/s, and road points are 0.5 m apart, so 1 sample takes 0.5/(100/3.6) = 0.018 s
            M(i,i2,i3,4)=range(Interpolate.cond(i2).lateral_dev(i,segment),'omitnan'); % lateral position range (m)
            M(i,i2,i3,5)=mean(Interpolate.cond(i2).Gain(i,segment),'omitnan'); % high SG (between 0 and 1)
        end
    end
end

M(:,:,3,7)=questionnaires.NASATLX.Overall; % NASA TLX overall (0 to 100)
M(:,:,3,8)=questionnaires.Ranking; % preference rank (1 to 4)
counter=0;

for i=1:24 % loop over 24 participants
    for i2=1:4 % loop over the 4 experimental conditions
        order=find(Order(i,:)==i2); % for example, if i = 24 (participant 24) and i2 = 3 (MI), order = 4 (i.e., condition was presented fourth)
        Mo(i,order,:,:)=M(i,i2,:,:); % mean per dependent measure and order of presentation
    end
end

[ST,STo]=deal(NaN(24,20));
for i3=1:3 % loop over 3 segments
    for i4=1:size(M,4) % number of measures
        m=M(:,:,i3,i4);
        [~,p(1),~,t]=ttest(m(:,1),m(:,2));dv(1)=t.tstat/sqrt(24); % p, t, and dz for FL vs. FH
        [~,p(2),~,t]=ttest(m(:,1),m(:,3));dv(2)=t.tstat/sqrt(24); % p, t, and dz for FL vs. MI
        [~,p(3),~,t]=ttest(m(:,1),m(:,4));dv(3)=t.tstat/sqrt(24); % p, t, and dz for FL vs. DI
        [~,p(4),~,t]=ttest(m(:,2),m(:,3));dv(4)=t.tstat/sqrt(24); % p, t, and dz for FH vs. MI
        [~,p(5),~,t]=ttest(m(:,2),m(:,4));dv(5)=t.tstat/sqrt(24); % p, t, and dz for FH vs. DI
        [~,p(6),~,t]=ttest(m(:,3),m(:,4));dv(6)=t.tstat/sqrt(24); % p, t, and dz for MI vs. DI
        counter=counter+1;
        ST(counter,:)=[mean(m) std(m) dv p]; % mean, SD, dz, and p

        m=Mo(:,:,i3,i4);
        [~,p(1),~,t]=ttest(m(:,1),m(:,2));dv(1)=t.tstat/sqrt(24); % p, t, and dz for FL vs. FH
        [~,p(2),~,t]=ttest(m(:,1),m(:,3));dv(2)=t.tstat/sqrt(24); % p, t, and dz for FL vs. MI
        [~,p(3),~,t]=ttest(m(:,1),m(:,4));dv(3)=t.tstat/sqrt(24); % p, t, and dz for FL vs. DI
        [~,p(4),~,t]=ttest(m(:,2),m(:,3));dv(4)=t.tstat/sqrt(24); % p, t, and dz for FH vs. MI
        [~,p(5),~,t]=ttest(m(:,2),m(:,4));dv(5)=t.tstat/sqrt(24); % p, t, and dz for FH vs. DI
        [~,p(6),~,t]=ttest(m(:,3),m(:,4));dv(6)=t.tstat/sqrt(24); % p, t, and dz for MI vs. DI
        STo(counter,:)=[mean(m) std(m) dv p]; % mean, SD, dz, and p
    end
end

disp('Table 1. Means (M), standard deviations (SD), and effect sizes (dz) per dependent measure and experimental condition.')
disp(ST)
disp('Table S1. Means (M), standard deviations (SD), and effect sizes (dz) per dependent measure and order of presentation (1 = first trial, 2 = second trial, 3 = third trial, 4 = fourth trial).')
disp(STo)
%%
Interpolate.mean_SWA_FL = mean(Interpolate.cond(1).SWA, 'omitnan'); % mean steering wheel angle
Interpolate.mean_SWA_FH = mean(Interpolate.cond(2).SWA, 'omitnan');
Interpolate.mean_SWA_MI = mean(Interpolate.cond(3).SWA, 'omitnan');
Interpolate.mean_SWA_DI = mean(Interpolate.cond(4).SWA, 'omitnan');

Interpolate.lateral_dev_FL = mean(((Interpolate.cond(1).lateral_dev)), 'omitnan'); % mean lateral position (m)
Interpolate.lateral_dev_FH = mean(((Interpolate.cond(2).lateral_dev)), 'omitnan');
Interpolate.lateral_dev_MI = mean(((Interpolate.cond(3).lateral_dev)), 'omitnan');
Interpolate.lateral_dev_DI = mean(((Interpolate.cond(4).lateral_dev)), 'omitnan');

abs_lateral_dev_vel_FL = mean(abs((1/0.018)*diff(Interpolate.cond(1).lateral_dev')),2, 'omitnan'); % absolute lateral velocity (m/s) (ego-vehicle speed is 100 km/h; distance between samples is 0.5 m, so 1 sample takes 0.018 s)
abs_lateral_dev_vel_FH = mean(abs((1/0.018)*diff(Interpolate.cond(2).lateral_dev')),2, 'omitnan');
abs_lateral_dev_vel_MI = mean(abs((1/0.018)*diff(Interpolate.cond(3).lateral_dev')),2, 'omitnan');
abs_lateral_dev_vel_DI = mean(abs((1/0.018)*diff(Interpolate.cond(4).lateral_dev')),2, 'omitnan');

Gain_FL =  sum(Interpolate.cond(1).Gain, 'omitnan'); % number of participants driving with high steering gain
Gain_FH =  sum(Interpolate.cond(2).Gain, 'omitnan');
Gain_MI =  sum(Interpolate.cond(3).Gain, 'omitnan');
Gain_DI =  sum(Interpolate.cond(4).Gain, 'omitnan');

%% Figure 2. The mean steering wheel angle (second panel), mean lateral position with respect to the center of the right lane (third panel), and mean absolute lateral velocity (fourth panel) for the fixed low (FL) and fixed high (FH) conditions. Positive values indicate a left curve, steering, and lateral movement. The first (top) panel shows traffic density, road curvature, and three horizontal line segments that demarcate the three segments used in the analysis: overtaking, straight, and curves.
figure
subplot(4,1,1)
colororder([color_FL;0 0 0])
yyaxis left
plot(DENSITY(1:35, 4), DENSITY(1:35, 2),'-','color',color_FL);hold on;box off;grid on;ax=gca;ax.GridColor = [0 0 0];
plot([1963 3464],[36 36],'k-x','Linewidth',30,'markerfacecolor',[0 0 0],'markersize',10)
plot([5500 9618],[36 36],'k-x','Linewidth',30,'markerfacecolor',[0.5 0.5 0.5],'markersize',10)
plot([9758 14345],[36 36],'k-x','Linewidth',30,'markerfacecolor',[1 1 1],'markersize',10)
ylabel({'Traffic density';'(cars/km)'});ylim([-40 40]);yyaxis right
plot(ROAD_DISTANCE, CURVATURE,'color','k');hold on
ylabel('Curvature (1/m)')
axis([0 14400 -0.012 0.012])
set(gca,'xtick',0:1000:15000)
set(gca,'XTickLabel',[]);

clear k
subplot(4,1,2)
k(2) = plot(Roadpoints, Interpolate.mean_SWA_FH, 'Color', color_FH);
hold on;box off;grid on;ax=gca;ax.GridColor = [0 0 0];
k(1) = plot(Roadpoints, Interpolate.mean_SWA_FL, 'Color', color_FL);
axis([0 14400 -70 70])
set(gca,'xtick',0:1000:15000)
ylabel({'Steering wheel';'angle (deg)'})
legend(k, 'Fixed low (FL)', 'Fixed high (FH)','Orientation','horizontal','location','north');
set(gca,'XTickLabel',[]);

clear k
subplot(4,1,3)
k(2) = plot(Roadpoints, Interpolate.lateral_dev_FH, 'Color', color_FH);
hold on;box off;grid on;ax=gca;ax.GridColor = [0 0 0];
k(1) = plot(Roadpoints, Interpolate.lateral_dev_FL, 'Color', color_FL);
legend(k, 'Fixed low (FL)', 'Fixed high (FH)','Orientation','horizontal','location','north');
xlim([0 14400])
set(gca,'xtick',0:1000:15000)
ylabel({'Lateral position';'(m)'});
set(gca,'XTickLabel',[]);

clear k
subplot(4,1,4)
k(2) = plot(Roadpoints(1:end-1), abs_lateral_dev_vel_FH, 'Color', color_FH);
hold on;box off;grid on;ax=gca;ax.GridColor = [0 0 0];
k(1) = plot(Roadpoints(1:end-1), abs_lateral_dev_vel_FL, 'Color', color_FL);
legend(k, 'Fixed low (FL)', 'Fixed high (FH)','Orientation','horizontal','location','north');
xlim([0 14400])
set(gca,'xtick',0:1000:15000)
ylabel({'Abs. lateral';'velocity (m/s)'});xlabel('Distance (m)')

set(findall(gcf,'-property','LineWidth'), 'LineWidth', 1.3)
set(findall(gcf,'-property','FontSize'),'FontSize',20)
set(gcf,'WindowState','maximized');

%% Figure 3. The mean steering wheel angle (second panel), mean absolute lateral velocity (third panel), and the number of participants driving with high steering gain (SG) (fourth panel) for the machine-initiated (MI) and driver-initiated (DI) conditions. Positive values indicate a left curve, steering, and lateral movement. The first (top) panel shows traffic density, road curvature, and three horizontal line segments that demarcate the three segments used in the analysis: overtaking, straight, and curves.
figure
subplot(4,1,1)
colororder([color_MI;0 0 0])
yyaxis left
plot(DENSITY(1:35, 4), DENSITY(1:35, 2),'-','color',color_MI);hold on;box off;grid on;ax=gca;ax.GridColor = [0 0 0];
plot([1963 3464],[36 36],'k-x','Linewidth',30,'markerfacecolor',[0 0 0],'markersize',10)
plot([5500 9618],[36 36],'k-x','Linewidth',30,'markerfacecolor',[0.5 0.5 0.5],'markersize',10)
plot([9758 14345],[36 36],'k-x','Linewidth',30,'markerfacecolor',[1 1 1],'markersize',10)
ylabel({'Traffic density';'(cars/km)'});ylim([-40 40]);yyaxis right
plot(ROAD_DISTANCE, CURVATURE,'color','k');hold on
ylabel('Curvature (1/m)')
set(gca,'xtick',0:1000:15000)
axis([0 14400 -0.012 0.012])
set(gca,'XTickLabel',[]);

clear k
subplot(4,1,2)
k(2) = plot(Roadpoints, Interpolate.mean_SWA_DI, 'Color', color_DI);
hold on;box off;grid on;ax=gca;ax.GridColor = [0 0 0];
k(1) = plot(Roadpoints, Interpolate.mean_SWA_MI, 'Color', color_MI);
set(gca,'xtick',0:1000:15000)
axis([0 14400 -70 70])
ylabel({'Steering wheel';'angle (deg)'})
legend(k, 'Machine-initiated (MI)', 'Driver-initiated (DI)','Orientation','horizontal','location','north')
set(gca,'XTickLabel',[]);

clear k
subplot(4,1,3)
k(2) = plot(Roadpoints(1:end-1), abs_lateral_dev_vel_DI, 'Color', color_DI);
hold on;box off;grid on;ax=gca;ax.GridColor = [0 0 0];
k(1) = plot(Roadpoints(1:end-1), abs_lateral_dev_vel_MI, 'Color', color_MI);
set(gca,'xtick',0:1000:15000)
xlim([0 14400])
ylabel({'Abs. lateral';'velocity (m/s)'});%xlabel('Distance (m)')
legend(k, 'Machine-initiated (MI)', 'Driver-initiated (DI)','Orientation','horizontal','location','north')
set(gca,'XTickLabel',[]);

clear k
subplot(4,1,4)
k(2) = plot(Roadpoints, Gain_DI, 'Color', color_DI);
hold on;box off;grid on;ax=gca;ax.GridColor = [0 0 0];
k(1) = plot(Roadpoints, Gain_MI, 'Color', color_MI);
set(gca,'xtick',0:1000:15000)
xlim([0 14400])
ylim([0 24])
ylabel({'Number of';'participants'});xlabel('Distance (m)')
legend(k, 'Machine-initiated (MI)', 'Driver-initiated (DI)','location','southeast')

set(findall(gcf,'-property','LineWidth'), 'LineWidth', 1.3)
set(findall(gcf,'-property','FontSize'),'FontSize',20)
set(gcf,'WindowState','maximized');

%% Figure 5. Ranking of the steering systems, with 1 representing the most preferred and 4 the least preferred.
figure('color','w','position',[488   342   560   420]);
x = [1 2 3 4];
y = [sum(questionnaires.Ranking == 1);
    sum(questionnaires.Ranking == 2);
    sum(questionnaires.Ranking == 3);
    sum(questionnaires.Ranking == 4)];

a = bar(x,y',0.4,'stacked','linewidth',1);box off
a(4).FaceColor = [0.1 0.1 0.1];
a(3).FaceColor = [0.5 0.5 0.5];
a(2).FaceColor = [0.8 0.8 0.8];
a(1).FaceColor = [1 1 1];

xticks([1 2 3 4])
xlim([0.5 4.5])
ylim([0 24])
xticklabels({'Fixed low (FL)' 'Fixed high (FH)' 'Machine-initiated (MI)' 'Driver-initiated (DI)'})
yticks(0:3:24)
grid on
ylabel({'Number of participants'})
legend(a,{'1','2','3','4'},'Orientation','horizontal','Position',[0.427,0.937,0.1847,0.0548])

set(findall(gcf,'-property','FontSize'),'FontSize',24)
set(gcf,'pos',[410 218 1090 769])

%% Figure 4. Mean absolute lateral velocity and lateral position range for the fixed high (FH) vs. fixed low (FL) conditions, and for the driver-initiated (DI) vs. machine-initiated (MI) conditions, for the curve-driving segment. Each marker represents a participant. The diagonal line is the line of unity.
temp=M(:,:,3,3); % Segment 3 is the curve segment; Measure 3 is the mean absolute lateral velocity

figure
subplot(2,2,1)
s=scatter(temp(:,1),temp(:,2),80,'markerfacecolor',[0 0 0],'markeredgecolor',[0 0 0]);hold on;grid on
s.MarkerFaceAlpha = .3;
hold on
plot([0 20],[0 20],'k')
axis equal
set(gca,'xlim',[0.2 0.7])
set(gca,'ylim',[0.2 0.7])
xlabel('Mean abs. lateral velocity - FL (m/s)')
ylabel('Mean abs. lateral velocity - FH (m/s)')

subplot(2,2,3)
s=scatter(temp(:,3),temp(:,4),80,'markerfacecolor',[0 0 0],'markeredgecolor',[0 0 0]);hold on;grid on
s.MarkerFaceAlpha = .3;
hold on
plot([0 20],[0 20],'k')
axis equal
set(gca,'xlim',[0.2 0.7])
set(gca,'ylim',[0.2 0.7])
xlabel('Mean abs. lateral velocity - MI (m/s)')
ylabel('Mean abs. lateral velocity - DI (m/s)')

temp=M(:,:,3,4); % Segment 3 is the curve segment; Measure 4 is the lateral position range

subplot(2,2,2)
s=scatter(temp(:,1),temp(:,2),80,'markerfacecolor',[0 0 0],'markeredgecolor',[0 0 0]);hold on;grid on
s.MarkerFaceAlpha = .3;
hold on
plot([0 20],[0 20],'k')
axis equal
set(gca,'xlim',[1.5 6])
set(gca,'ylim',[1.5 6])
xlabel('Lateral position range - FL (m)')
ylabel('Lateral position range - FH (m)')

subplot(2,2,4)
s=scatter(temp(:,3),temp(:,4),80,'markerfacecolor',[0 0 0],'markeredgecolor',[0 0 0]);hold on;grid on
s.MarkerFaceAlpha = .3;
hold on
plot([0 20],[0 20],'k')
axis equal
set(gca,'xlim',[1.5 6])
set(gca,'ylim',[1.5 6])
xlabel('Lateral position range - MI (m)')
ylabel('Lateral position range - DI (m)')
h=findobj('FontName','Helvetica');set(h,'FontSize',12,'Fontname','Arial')
set(gcf,'position',[502.333333333333 403 737.666666666667 695])

%% Table S2. Means (M), standard deviations (SD), and correlations between the proportion of time driving with the high SG settings in the driver-initiated (DI) condition, and preference rankings for the four conditions (n = 24).
v1 = M(:,:,3,8); % 24 x 4 matrix of preference ranks (1 = most preferred). FL, FH, MI, DI
v2 = squeeze(M(:,4,:,5)); % 24 x 3 matrix of Mean SG for the DI condition. Overtaking, Straight, Curves
disp('Mean and SD')
disp(round([mean([v1 v2])' std([v1 v2])'],2))
disp(round(corr([v1 v2]),2))
